/*
 *    $Id: files.c,v 1.1.1.1 1998/04/21 02:21:26 hideki Exp $
 *    This file is a part of the Workplace.
 *    Copyright (C) 1997, 1998  Hideki Fujimoto.
 */

#include <gtk/gtk.h>
#include <ctype.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <unistd.h>
#include "files.h"

#define MAX_SIZE 5120

/* Return an int indicating the result of comparing two longs. */
#define longdiff(a, b) ((a) - (b))

static FileList *arry2list (Fileinfo  *files[], int index);
static int sort_by_name (const Fileinfo **file1, const Fileinfo **file2);
static int sort_by_time (const Fileinfo **file1, const Fileinfo **file2);
static int sort_by_size (const Fileinfo **file1, const Fileinfo **file2);

extern DisplayType Display_type;

/* This func initalize a FileList */
FileList *
init_files(const char *path, enum sorttype stype)
{
	Fileinfo      *files[MAX_SIZE];
	struct dirent *dirent;
        struct stat   *st;
	DIR           *dir;
	char          *name;
	char          *abbr;
	char          *pathname;
	char          *ppath;
	int           index = 0;
	int           (*func)() = NULL;
	
	ppath = (char *) g_malloc(strlen(path)+1);
	strcpy(ppath, path);

	dir = opendir(ppath);
	if (!dir) {
		fprintf(stderr,"opendir(): err <%s>\n",ppath);
		return NULL;
	}

	while (1) {
		dirent = readdir(dir);
		if (!dirent){
			break;
		}	

		/* clean string */
		name = (char *) g_malloc(strlen(dirent->d_name)+1);
		if (!name){
			fprintf(stderr,"init_files: g_malloc err\n");
			break;
		}
		else {
			strcpy(name, dirent->d_name);
		}
		if (strcasecmp(name,".") == 0 || strcasecmp(name,"..") == 0){
			g_free(name);
			name = NULL;
		}
		else {
			/* make abosolute pathname */
			pathname = (char *) g_malloc(strlen(path)+strlen(name)+1);
			if (!pathname){
				fprintf(stderr,"init_files: g_malloc err\n");
				break;
			}
			strcpy(pathname, ppath);
			strcat(pathname, name);

			st = (struct stat *) g_malloc(sizeof(struct stat));
			/* get  information of a file represented by pathname */
			if (stat(pathname, st) == -1){
				if (lstat(pathname, st) == -1) {
					g_free(pathname);
					g_free(st);
					continue;
				}
			}

			/* check permission */
			files[index] =  (Fileinfo *) g_malloc(sizeof(Fileinfo));
			if (!files[index]){
				fprintf(stderr,"init_files: g_malloc err\n");
				break;
			}
			
			/* set path */
			files[index]->path = ppath;

			/* set filename */
			files[index]->name =  name;
			if (strlen(name)>18){
				abbr = (char *) g_malloc(19);
				if (!abbr){
					fprintf(stderr,"init_files: g_malloc err\n");
					gtk_exit(1);
				}
				strncpy(abbr, name, 16);
				*(abbr+16) = '\0';
				strcat(abbr, "...");
				files[index]->name_abbr = abbr;
			}
			else {
				files[index]->name_abbr = NULL;
			}
			
			/* set stat */
			files[index]->stat = st;
			
			/* set NULL */
			files[index]->day_abbr = NULL;

			/* set st */
			if (S_ISREG(st->st_mode)){
				files[index]->filetype = NORMAL;
			}
			else if (S_ISLNK(st->st_mode)){
				files[index]->filetype = S_LINK;
			}
			else if (S_ISDIR(st->st_mode)){
				files[index]->filetype = DIRECTORY;
			}
			else if (S_ISCHR(st->st_mode)){
				files[index]->filetype = NONE;
			}
			else if (S_ISBLK(st->st_mode)){
				files[index]->filetype = NONE;
			}
			else {
				files[index]->filetype = NONE;
			}
			g_free(pathname);
			index++;
			if (index == MAX_SIZE){
				fprintf(stderr,"file size over\n");
				break;
			}
		}
	}
	closedir(dir);

	/* no contain */
	if (index == 0){
		return NULL;
	}

	if (stype == BY_NAME) {
		func = sort_by_name;
	}
	else if (stype == BY_SIZE) {
		func = sort_by_size;
	}
	else if (stype == BY_TIME) {
		func = sort_by_time;
	}
		
	qsort(files, index, sizeof(Fileinfo *), func);

	return arry2list(files, index);
}

static FileList *
arry2list (Fileinfo  *files[], int index)
{
	FileList      *top;
	FileList      *list;
	FileList      *buf;
	register int  i;

	/* restore */
	list = (FileList *) g_malloc(sizeof(FileList));
	list->prev = NULL;
	list->next = NULL;
	list->icon = NULL;

	/* set null */
	list->size = NULL;
	list->time = NULL;

	list->info= files[0];
	top = list;


	for (i=1;i<index;i++) {
		buf = list;
		list->next = (FileList *) g_malloc(sizeof(FileList));
		list = list->next;
		list->prev = buf;
		list->next = NULL;
		list->info = files[i];
		list->info->parent = list;

                /* set null */
		list->size = NULL;
		list->time = NULL;
	}

	return top;
}

FileList *
sort_files (FileList  *list, enum sorttype stype)
{
	FileList      *garbage;
	Fileinfo      *files[MAX_SIZE];
	int           index;
	int           (*func)() = NULL;
	register int  i = 0;

	while (1) {
		if (!list) {
			break;
		}
		if (list->size) {
			g_free(list->size);
			list->size = NULL;
		}
		if (list->time) {
			g_free(list->time);
			list->time = NULL;
		}

		files[i] = list->info;
		i++;
		
		/* set garbage */
		garbage = list;
		
		/* next */
		list = list->next;

		/* clear garbage */
		g_free(garbage);
	}
	
	index = i;

	if (index == 0){
		return NULL;
	}

	if (stype == BY_NAME) {
		func = sort_by_name;
	}
	else if (stype == BY_SIZE) {
		func = sort_by_size;
	}
	else if (stype == BY_TIME) {
		func = sort_by_time;
	}
		
	qsort(files, index, sizeof(Fileinfo *), func);

	return arry2list(files, index);
}

void 
clear_files(FileList *file)
{
	FileList   *garbage;

	while (1) {
		if (!file){
			break;
		}
		else {
			if (file->info){
				if (file->info->name) {
					g_free(file->info->name);
					file->info->name = NULL;
				}
				if (file->info->name_abbr) {
					g_free(file->info->name_abbr);
					file->info->name_abbr = NULL;
				}
				if (file->info->day_abbr) {
					g_free(file->info->day_abbr);
					file->info->day_abbr = NULL;
				}
				if (file->info->stat) {
					g_free(file->info->stat);
					file->info->stat = NULL;
				}
				g_free(file->info);
				file->info = NULL;
			}

			/* destroy widget */
			if (Display_type == ICON) {
				if (file->icon) {
					//gtk_widget_unrealize(file->image);
					//gtk_grab_remove(file->image);
					//gtk_object_remove_data(GTK_OBJECT(file->image),
					//		       "user_data"); 
					//gtk_signal_handlers_destroy (GTK_OBJECT(file->icon));
					//gtk_widget_hide(file->icon);
					//gtk_widget_destroy(file->icon);
					file->icon = NULL;
					file->image = NULL;
				}
			}
			else if (Display_type == LIST) {
				if (file->size) {
					g_free(file->size);
				}
				if (file->time) {
					g_free(file->time);
				}
			}
			
			garbage = file;
			file = file->next;
			if (garbage){
				g_free(garbage);
				garbage = NULL;
			}
		}
	}
}


static int 
sort_by_name(const Fileinfo **file1, const Fileinfo **file2)
{
	 return strcmp((*file1)->name, (*file2)->name);
}

static int 
sort_by_time(const Fileinfo **file1, const Fileinfo **file2)
{
	 return longdiff((*file2)->stat->st_ctime, (*file1)->stat->st_ctime);
}

static int 
sort_by_size(const Fileinfo **file1, const Fileinfo **file2)
{
	 return longdiff((*file2)->stat->st_size, (*file1)->stat->st_size);
}





